/*
 * Copyright (c) 2010, Nieko Maatjes
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. This program may not be used for commercial purposes without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY NIEKO MAATJES ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL NIEKO MAATJES BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "main.h"

#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MIN(a, b) ((a) < (b) ? (a) : (b))

PSP_MODULE_INFO("PSP Guitar", PSP_MODULE_USER, 0, 1); // version: major, minor
PSP_MAIN_THREAD_ATTR(PSP_THREAD_ATTR_USER | PSP_THREAD_ATTR_VFPU);
PSP_HEAP_SIZE_KB(20000); // 20 MB for malloc()

short mutexFull = 0;

// AUDIO
#define SAMPLE_RATE (44100)
#define SAMPLE_SIZE (1024)
static SoundBuffer *outBuffer;
static Chord *chord;
static short octave;
typedef struct { short l, r; } sample_t;

// CTRL
SceCtrlData pad, oldPad;

// EXTERN
extern unsigned int __attribute__((aligned(16))) guList[262144];

int main()
{
    setupExitCallback();
    graphicsInit();

    pspAudioInit();
    pspAudioSetChannelCallback(0, audioCallback, NULL);

    outBuffer = newSoundBuffer(SAMPLE_SIZE, 0, "");
    chord = newChord();
    octave = 4;

    int decreaseOctave = 0; int increaseOctave = 0;
    while (1)
    {
        oldPad.Buttons = pad.Buttons;
        sceCtrlReadBufferPositive(&pad, 1);

        // Start graphics
        sceGuStart(GU_DIRECT, guList);
        sceGuClearColor(0);
        sceGuClear(GU_COLOR_BUFFER_BIT);

        // Mutual exclusion, progress and bounded waiting :)
        while (mutexFull)
        { sceKernelDelayThread(1e2); }
        {
            // Buttons changed
            if ((pad.Buttons ^ oldPad.Buttons))
            {
                char note[4] = { 0, 0, 0, 0 };

                // Flat/sharp note
                // Interrupt octave change
                if ((pad.Buttons & PSP_CTRL_LTRIGGER)
                 && (pad.Buttons ^ PSP_CTRL_LTRIGGER))
                { decreaseOctave = 0; note[2] = 'b'; }
                if ((pad.Buttons & PSP_CTRL_RTRIGGER)
                 && (pad.Buttons ^ PSP_CTRL_RTRIGGER))
                { increaseOctave = 0; note[2] = '#'; }

                // Change octave
                {
                    // L not pressed, now is only key pressed
                    if ((oldPad.Buttons & PSP_CTRL_LTRIGGER) == 0
                     && (pad.Buttons ^ PSP_CTRL_LTRIGGER) == 0)
                    { decreaseOctave = 1; }
                    // Only L pressed, not anymore
                    if ((oldPad.Buttons ^ PSP_CTRL_LTRIGGER) == 0
                     && (pad.Buttons & PSP_CTRL_LTRIGGER) == 0
                     && decreaseOctave)
                    { octave--; }

                    // R not pressed, now is only key pressed
                    if ((oldPad.Buttons & PSP_CTRL_RTRIGGER) == 0
                     && (pad.Buttons ^ PSP_CTRL_RTRIGGER) == 0)
                    { increaseOctave = 1; }
                    // Only R pressed, not anymore
                    if ((oldPad.Buttons ^ PSP_CTRL_RTRIGGER) == 0
                     && (pad.Buttons & PSP_CTRL_RTRIGGER) == 0
                     && increaseOctave)
                    { octave++; }

                    if (octave < 2) { octave = 2; }
                    if (octave > 6) { octave = 6; }
                }

                note[1] = octave+48; // 0..9 => 48..57

                // Enable multiple notes at once
                if (pad.Buttons & PSP_CTRL_TRIANGLE) { note[0] = 'B'; pluck(note); }
                if (pad.Buttons & PSP_CTRL_SQUARE)   { note[0] = 'A'; pluck(note); }
                if (pad.Buttons & PSP_CTRL_CIRCLE)   { note[0] = 'G'; pluck(note); }
                if (pad.Buttons & PSP_CTRL_CROSS)    { note[0] = 'F'; pluck(note); }
                if (pad.Buttons & PSP_CTRL_UP)       { note[0] = 'E'; pluck(note); }
                if (pad.Buttons & PSP_CTRL_LEFT)     { note[0] = 'D'; pluck(note); }
                if (pad.Buttons & PSP_CTRL_RIGHT)    { note[0] = 'C'; pluck(note); }
            }

            tic();
        }
        mutexFull = 1;

        printInfo();

        displayWave();

        graphicsFlush();
    }

    die(NULL);

    return 0;
}

// Start new note, fill with white noise
// http://en.wikipedia.org/wiki/Karplus-Strong
void pluck(char *note)
{
    int frequency = noteToFrequency(note);
    SoundBuffer *newNote = addSoundBufferToChord(chord, SAMPLE_RATE/frequency, frequency, note);
    if (newNote == NULL)
    { return; }

    time_t now = time(NULL);
    srand(now);

    int i = newNote->size;
    while (i-- > 0)
    {
        // Snare
        newNote->samples->value = (rand()%65536)-32768;

        newNote->samples = newNote->samples->next;
    }
}

int noteToFrequency(char *note)
{
                       // A  B   C   D   E   F   G
    int octaveOffset[] = {0, 2, -9, -7, -5, -4, -2}; // 2**(n/12) increments

    short half = 0;
    if (note[2] == 'b') { half = -1; }
    if (note[2] == '#') { half =  1; }

    int noteFactor = octaveOffset[toupper(note[0]-65)]+half;
    int baseFrequency = 55<<(atoi(note+1)-1); // 55Hz is C1..B1

    return roundf(baseFrequency*powf(2.0f, 1.0f*noteFactor/12));
}

void tic()
{
    ShortList *samples = outBuffer->samples;
    // Temporary buffer of ints to handle overflows, which might cancel each other out
    int temp[SAMPLE_SIZE];

    // Empty buffers
    int i;
    for (i = 0; i < SAMPLE_SIZE; i++)
    {
        temp[i] = 0;
        samples->value = 0;
        samples = samples->next;
    }

    // Go through all SoundBuffers in the Chord
    SoundBufferList *current = chord->buffers;
    while (current != NULL)
    {
        for (i = 0; i < SAMPLE_SIZE; i++)
        {
            // Attenuate current sample
            current->value->samples->value = current->value->decay
                                            *(0.5f*current->value->samples->value
                                             +0.5f*current->value->samples->next->value);

            temp[i] += current->value->samples->value;

            // Move to next item
            current->value->samples = current->value->samples->next;
        }

        // Remove SoundBuffer when silent
        if ((current->value->samples->value
           | current->value->samples->next->value
           | current->value->samples->next->next->value
           | current->value->samples->next->next->next->value)
          == 0)
        {
            deleteSoundBufferFromChord(chord, current->value);
        }

        current = current->next;
    }

    // Limit to short (16 bits)
    for (i = 0; i < SAMPLE_SIZE; i++)
    {
        if (temp[i] > 32767)
        { samples->value = 32767; }
        else if (temp[i] < -32768)
        { samples->value = -32768; }
        else
        { samples->value = temp[i]; }
        samples = samples->next;
    }
}

void printInfo()
{
    char notes[128] = "";

    SoundBufferList *current = chord->buffers;
    while (current != NULL)
    {
        if (strlen(notes) != 0)
        { strcat(notes, ", "); }
        strcat(notes, current->value->note);

        current = current->next;
    }
    pspDebugScreenSetXY(0, 0);
    printf("Notes (%d): %s\n", octave, notes);
}

void displayWave()
{
    float sx, sy, dx, dy;

    ShortList *samples = outBuffer->samples;
    int i;
    for (i = 0; i < SAMPLE_SIZE-2; i += 2)
    {
        sx = ((float)i) * (SCREEN_WIDTH-1) / (SAMPLE_SIZE-1);
        sy = SCREEN_HEIGHT - (((float)samples->next->next->value)+32768) * (SCREEN_HEIGHT-1) / (65535);
        dx = ((float)i-2) * (SCREEN_WIDTH-1) / (SAMPLE_SIZE-1);
        dy = SCREEN_HEIGHT - (((float)samples->value)+32768) * (SCREEN_HEIGHT-1) / (65535);
        drawLine(sx, sy, dx, dy, 0xFFFFFFFF);

        samples = samples->next->next;
    }
}

// *buf = 16-bit stereo (short)
void audioCallback(void *buf, unsigned int length, void *userdata)
{
    ShortList *samples = outBuffer->samples;

    while (!mutexFull)
    { sceKernelDelayThread(1e2); }
    {
        sample_t *out = (sample_t *)buf;
        int i;
        for (i = 0; i < SAMPLE_SIZE; i++)
        {
            out[i].l = samples->value;
            out[i].r = samples->value;
            samples = samples->next;
        }
    }
    mutexFull = 0;
}
