/*
 * Copyright (c) 2010, Nieko Maatjes
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. This program may not be used for commercial purposes without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY NIEKO MAATJES ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL NIEKO MAATJES BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "main.h"

PSP_MODULE_INFO("PSP Tone Generator", PSP_MODULE_USER, 0, 2); // version: major, minor
PSP_MAIN_THREAD_ATTR(THREAD_ATTR_USER | THREAD_ATTR_VFPU);

// AUDIO
static short buffer[2][SAMPLE_SIZE];
int curTone = 0;
int activeToneCount[2] = {0, 0};
int enabledTones[2][FREQUENCIES];
float curFrequencies[2][FREQUENCIES];
float posFrequencies[2][FREQUENCIES];
typedef struct { short l, r; } sample_t;

// CTRL
SceCtrlData pad, oldPad;

// EXTERN
extern unsigned int __attribute__((aligned(16))) guList[262144];

// OTHER
short waveType = 0; // 0 = red+blue, 1 = white (summed), 2 = both

int main()
{
    setupExitCallback();
    graphicsInit();

    pspAudioInit();
    pspAudioSetChannelCallback(0, audioCallback, NULL);

    int i, j;
    for (i = 0; i < 2; i++)
    {
        for (j = 0; j < FREQUENCIES; j++)
        {
            curFrequencies[i][j] = 440.0f;
            enabledTones[i][j] = 0;
            posFrequencies[i][j] = 0.0f;
        }
    }

    while (1)
    {
        checkControls();

        // Start graphics
        sceGuStart(GU_DIRECT, guList);
        sceGuClearColor(0);
        sceGuClear(GU_COLOR_BUFFER_BIT);

        printInfo();

        // Audio callback calls tone generator

        displayWave(buffer);

        graphicsFlush();
    }

    die(NULL);

    return 0;
}

// XXX: makes use of [0][curTone], treating the
// multi-dimensional array as if it were flattened.
void checkControls()
{
    oldPad.Buttons = pad.Buttons;
    sceCtrlReadBufferPositive(&pad, 1);

    // Enable/disable tone
    if ((pad.Buttons ^ oldPad.Buttons)
     && (pad.Buttons & PSP_CTRL_CROSS))
    {
        enabledTones[0][curTone] ^= 1;
        activeToneCount[curTone/FREQUENCIES] += enabledTones[0][curTone] == 1 ? 1 : -1;
    }

    // Change tone
    if ((pad.Buttons ^ oldPad.Buttons)
     && (pad.Buttons & PSP_CTRL_UP))
    {
        curTone--;
    }
    else if ((pad.Buttons ^ oldPad.Buttons)
          && (pad.Buttons & PSP_CTRL_DOWN))
    {
        curTone++;
    }
    else if ((pad.Buttons ^ oldPad.Buttons)
          && (pad.Buttons & PSP_CTRL_LEFT))
    {
        curTone-=FREQUENCIES;
    }
    else if ((pad.Buttons ^ oldPad.Buttons)
          && (pad.Buttons & PSP_CTRL_RIGHT))
    {
        curTone+=FREQUENCIES;
    }

    // Limit tones
    if (curTone < 0)
    { curTone += 2*FREQUENCIES; }
    else if (curTone >= 2*FREQUENCIES)
    { curTone -= 2*FREQUENCIES; }

    // Change current frequency
    int i, j;
    if (pad.Buttons & PSP_CTRL_RTRIGGER)
    {
        for (i = 0; i < 2*FREQUENCIES; i++)
        {
            if ((i == curTone)
             || (pad.Buttons & PSP_CTRL_CIRCLE))
            { curFrequencies[0][i] += frequencyDiff(&pad); }
        }
    }
    else if (pad.Buttons & PSP_CTRL_LTRIGGER)
    {
        for (i = 0; i < 2*FREQUENCIES; i++)
        {
            if ((i == curTone)
             || (pad.Buttons & PSP_CTRL_CIRCLE))
            { curFrequencies[0][i] -= frequencyDiff(&pad); }
        }
    }

    // Limit frequencies
    for (i = 0; i < 2; i++)
    {
        for (j = 0; j < FREQUENCIES; j++)
        {
            if (curFrequencies[i][j] < 20)    { curFrequencies[i][j] = 20; }
            if (curFrequencies[i][j] > 20480) { curFrequencies[i][j] = 20480; }
        }
    }

    // Show channels separately, together (summed), or both
    if ((pad.Buttons ^ oldPad.Buttons)
     && (pad.Buttons & PSP_CTRL_SELECT))
    {
        waveType++;
        if (waveType > 2)
        { waveType = 0; }
    }
}

void printInfo()
{
    pspDebugScreenSetXY(0, 0);
    pspDebugScreenSetTextColor(RGB(255, 255, 255));

    int numLines = 3;
    printf("X = enable/disable tone, L/R = change frequency\n");
    printf("O = change all frequencies, T/S = change frequency faster/slowlier\n");
    printf("Select = change wave display type (separate, summed, both)\n");

    int i, j;
    for (i = 0; i < 2; i++)
    {
        for (j = 0; j < FREQUENCIES; j++)
        {
            unsigned int color = enabledTones[i][j] ? RGB((1-i)*255, 0, i*255) : RGB((1-i)*127, 0, i*127);
            pspDebugScreenSetTextColor(color);
            pspDebugScreenSetXY(i*34, numLines+j);
            printf(" %s %d"
                  ,i*FREQUENCIES+j == curTone || (pad.Buttons & PSP_CTRL_CIRCLE)
                   ? "->"
                   : "  "
                  ,(int)curFrequencies[i][j]
                  );
        }
    }
}

void toneGenerator(short buffer[2][SAMPLE_SIZE])
{
    int i, j, k;
    for (i = 0; i < 2; i++)
    {
        for (j = 0; j < SAMPLE_SIZE; j++)
        {
            buffer[i][j] = 0;
            for (k = 0; k < FREQUENCIES; k++)
            {
                if (enabledTones[i][k])
                {
                    buffer[i][j] += (32767/activeToneCount[i])*sinf(GU_PI*2.0f*(int)curFrequencies[i][k]*posFrequencies[i][k]);
                    posFrequencies[i][k] += 1.0f/SAMPLE_RATE;
                }
            }
        }
    }

    double dummy;
    for (i = 0; i < 2; i++)
    {
        for (j = 0; j < FREQUENCIES; j++)
        {
            if (enabledTones[i][j])
            {
                if (posFrequencies[i][j] * (int)curFrequencies[i][j] >= 1.0f)
                { posFrequencies[i][j] = modf(posFrequencies[i][j] * (int)curFrequencies[i][j], &dummy) / (int)curFrequencies[i][j]; }
            }
        }
    }
}

void displayWave(short wave[2][SAMPLE_SIZE])
{
    float sx, sy, dx, dy, sumSx, sumSy, sumDx, sumDy;

    int i, j;
    for (i = 2; i < SAMPLE_SIZE; i += 2)
    {
        sumSx = sumSy = sumDx = sumDy = 0;
        for (j = 0; j < 2; j++)
        {
            sx = ((float)i) * (SCREEN_WIDTH-1) / (SAMPLE_SIZE-1);
            sy = SCREEN_HEIGHT - (((float)wave[j][i])+32768) * (SCREEN_HEIGHT-1) / (65535);
            dx = ((float)i-2) * (SCREEN_WIDTH-1) / (SAMPLE_SIZE-1);
            dy = SCREEN_HEIGHT - (((float)wave[j][i-2])+32768) * (SCREEN_HEIGHT-1) / (65535);
            sumSx += sx * 0.5;
            sumDx += dx * 0.5;
            sumSy += sy * 0.5;
            sumDy += dy * 0.5;
            if (waveType == 0 || waveType == 2)
            { drawLine(sx, sy, dx, dy, j == 0 ? 0xFF0000FF : 0XFFFF0000); }
        }
        if (waveType == 1 || waveType == 2)
        { drawLine(sumSx, sumSy, sumDx, sumDy, 0xFFFFFFFF); }
    }
}

float frequencyDiff(SceCtrlData *pad)
{
    if (pad->Buttons & PSP_CTRL_TRIANGLE)
    { return 20; }
    else if (pad->Buttons & PSP_CTRL_SQUARE)
    { return 0.1; }
    else
    { return 1; }
}

// *buf = 16-bit stereo (short)
void audioCallback(void *buf, unsigned int length, void *userdata)
{
    toneGenerator(buffer);

    sample_t *out = (sample_t *)buf;
    int i;
    for (i = 0; i < SAMPLE_SIZE; i++)
    {
        out[i].l = buffer[0][i];
        out[i].r = buffer[1][i];
    }
}
